// State Management
let currentState = 'idle';
let analysisData = null;
let analysisHistory = [];

// DOM Elements will be initialized on DOMContentLoaded
let idleState, analyzingState, resultsState, analyzeBtn, viewFullBtn, newAnalysisBtn, newAnalysisBtnPanel, panelOverlay, closePanel, historyBtn, historyModal, closeModal, settingsBtn, clearHistoryBtn, citationBtn, citationStatus;
let usageContainer, usageText, planBadge, usageBarFill, paywallContainer, upgradeBtn;


// Chrome is globally available in popup scripts


// Initialize
document.addEventListener('DOMContentLoaded', () => {
  initializeElements();
  loadHistoryFromStorage();
  setupEventListeners();
  setState('idle');
  updateUsageUI(); // Fetch and show usage on popup open
});

function initializeElements() {
  idleState = document.getElementById('idleState');
  analyzingState = document.getElementById('analyzingState');
  resultsState = document.getElementById('resultsState');
  analyzeBtn = document.getElementById('analyzeBtn');
  viewFullBtn = document.getElementById('viewFullBtn');
  newAnalysisBtn = document.getElementById('newAnalysisBtn');
  newAnalysisBtnPanel = document.getElementById('newAnalysisBtnPanel');
  panelOverlay = document.getElementById('panelOverlay');
  closePanel = document.getElementById('closePanel');
  historyBtn = document.getElementById('historyBtn');
  historyModal = document.getElementById('historyModal');
  closeModal = document.getElementById('closeModal');
  settingsBtn = document.getElementById('settingsBtn');
  clearHistoryBtn = document.getElementById('clearHistoryBtn');
  citationBtn = document.getElementById('citationBtn');
  citationStatus = document.getElementById('citationStatus');
  usageContainer = document.getElementById('usageContainer');
  usageText = document.getElementById('usageText');
  planBadge = document.getElementById('planBadge');
  usageBarFill = document.getElementById('usageBarFill');
  paywallContainer = document.getElementById('paywallContainer');
  upgradeBtn = document.getElementById('upgradeBtn');
}

// Event Listeners
function setupEventListeners() {
  analyzeBtn.addEventListener('click', handleAnalyze);
  viewFullBtn.addEventListener('click', showPanel);
  newAnalysisBtn.addEventListener('click', resetAnalysis);
  newAnalysisBtnPanel.addEventListener('click', resetAnalysis);
  closePanel.addEventListener('click', hidePanel);
  panelOverlay.addEventListener('click', hidePanel);
  historyBtn.addEventListener('click', showHistoryModal);
  closeModal.addEventListener('click', hideHistoryModal);
  settingsBtn.addEventListener('click', openSettings);
  clearHistoryBtn.addEventListener('click', clearHistory);
  citationBtn.addEventListener('click', generateCitation);
  upgradeBtn.addEventListener('click', () => {
    chrome.tabs.create({ url: 'https://aeoextension.com/pricing' });
  });

  // Prevent panel close on content click
  document.querySelector('.side-panel')?.addEventListener('click', (e) => {
    e.stopPropagation();
  });

  document.querySelector('.modal-content')?.addEventListener('click', (e) => {
    e.stopPropagation();
  });
}

// Analyze Handler
async function handleAnalyze() {
  console.log('[AEO] Analysis triggered');

  if (!analyzeBtn) return;

  // 1. Check usage limits first
  const usageResult = await checkUsage();
  if (!usageResult.success) {
    if (usageResult.error === 'LIMIT_EXCEEDED') {
      showPaywall(usageResult);
    } else if (usageResult.error === 'NOT_LOGGED_IN' && usageResult.used >= usageResult.limit) {
      // Only force login if the anonymous limit is also reached
      alert('You have reached the free limit. Please login or subscribe at aeoextension.com to continue.');
      chrome.tabs.create({ url: 'https://aeoextension.com/login' });
    } else {
      console.warn('[AEO] Usage check non-blocking error:', usageResult.error);
    }

    if (usageResult.error === 'LIMIT_EXCEEDED' || (usageResult.error === 'NOT_LOGGED_IN' && usageResult.used >= usageResult.limit)) {
      return;
    }
  }

  setState('analyzing');

  try {
    // Get current tab
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tabs || tabs.length === 0) {
      throw new Error('No active tab found');
    }
    const tab = tabs[0];

    // Check if we can analyze this page (not chrome:// or other restricted pages)
    if (tab.url.startsWith('chrome://') || tab.url.startsWith('edge://') || tab.url.startsWith('about:') || tab.url.startsWith('chrome-extension://')) {
      alert('This page cannot be analyzed. Please try a public website.');
      setState('idle');
      return;
    }

    console.log('[AEO] Requesting content from tab:', tab.id);

    // Try to send message to content script
    let response;
    try {
      response = await chrome.tabs.sendMessage(tab.id, { action: 'analyzeContent' });
    } catch (msgError) {
      // If content script is not loaded, try to inject it
      if (msgError.message.includes('Could not establish connection') || msgError.message.includes('Receiving end does not exist')) {
        console.log('[AEO] Content script missing, injecting...');
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          files: ['content.js']
        });
        // Small delay to let script initialize
        await new Promise(resolve => setTimeout(resolve, 200));
        // Retry message
        response = await chrome.tabs.sendMessage(tab.id, { action: 'analyzeContent' });
      } else {
        throw msgError;
      }
    }

    if (!response || !response.success) {
      throw new Error('Failed to get content from page');
    }

    console.log('[AEO] Content received, processing...');

    // Simulate a brief processing delay to show the "Analyzing" state
    setTimeout(() => {
      analysisData = generateAnalysisFromContent(response, tab.url);
      updateResults();
      setState('results');
      showPanel(); // Automatically show the full results panel
      console.log('[AEO] Analysis complete');
    }, 1000);

  } catch (error) {
    console.error('[AEO] Error during analysis:', error);
    alert('Analysis failed: ' + (error.message || 'Make sure the page is fully loaded and refresh it if needed.'));
    setState('idle');
  }
}

// Calculate AEO Score directly from technical signals
function calculateAEOScore(content) {
  const t = content.technical;
  let score = 0;

  // 1. H1 Integrity (20 pts) - Exactly one H1
  if (t.h1Count === 1) score += 20;
  else if (t.h1Count > 1) score += 10;

  // 2. Schema Detection (15 pts)
  if (t.hasSchema) score += 15;

  // 3. FAQ Presence (15 pts)
  if (t.hasFAQ) score += 15;

  // 4. Subheading Density (20 pts) - Goal is 5+ H2s
  if (t.h2Count >= 5) score += 20;
  else if (t.h2Count >= 2) score += 10;

  // 5. Intro Length (10 pts) - First para > 100 chars
  if (t.firstParaLength >= 100) score += 10;

  // 6. Content Depth (20 pts) - 5+ paragraphs
  if (content.paragraphs.length >= 5) score += 20;
  else if (content.paragraphs.length >= 2) score += 10;

  // Ensure score is between 20-100
  return Math.max(20, Math.min(100, score));
}

// Generate Analysis Data from Page Content
function generateAnalysisFromContent(content, tabUrl) {
  const t = content.technical;
  const aeoScore = calculateAEOScore(content);

  const success = [];
  const issues = [];
  const recommendations = [];

  // Success Areas
  if (t.h1Count === 1) success.push('Page has a single, focused H1 tag');
  if (t.h2Count >= 5) success.push('Excellent use of H2 subheadings (5+)');
  if (t.hasSchema) success.push('Schema markup detected');
  if (content.paragraphs.length >= 5) success.push('Strong content structure (5+ paragraphs)');
  if (t.hasFAQ) success.push('FAQ section detected and active');

  // Issues & Recommendations
  if (t.h1Count === 0) {
    issues.push('Missing H1 tag');
    recommendations.push('Add a descriptive H1 title to your page');
  } else if (t.h1Count > 1) {
    issues.push('Multiple H1 tags detected');
    recommendations.push('Reduce to a single H1 tag for better focus');
  }

  if (t.h2Count < 5) {
    issues.push(`Low subheading count (${t.h2Count})`);
    recommendations.push('Add more H2 subheadings to organize your content (aim for 5+)');
  }

  if (!t.hasFAQ) {
    issues.push('No FAQ section detected');
    recommendations.push('Add an FAQ section with common questions and answers about your topic');
  }

  if (t.firstParaLength < 100) {
    issues.push('First paragraph is too short');
    recommendations.push('Expand your introductory paragraph to at least 100 characters');
  }

  if (!t.hasSchema) {
    issues.push('No JSON-LD Schema detected');
    recommendations.push('Add FAQ or Article Schema to help AI engines understand your data');
  }

  if (!t.metaDesc) {
    issues.push('No meta description found');
    recommendations.push('Add a meta description to summarize your page content');
  }

  return {
    score: aeoScore,
    url: tabUrl,
    timestamp: new Date().getTime(),
    technical: {
      'H1 Tags': t.h1Count,
      'H2 Tags': t.h2Count,
      'Schema': t.hasSchema ? '✅ Found' : '❌ Missing',
      'FAQ Section': t.hasFAQ ? '✅ Found' : '❌ Missing',
      'First Para': `${t.firstParaLength} chars`
    },
    success: success.slice(0, 6),
    issues: issues.slice(0, 6),
    recommendations: recommendations.slice(0, 6),
    bestFragment: content.bestFragment
  };
}

// Update UI with Results
function updateResults() {
  document.getElementById('scoreValue').textContent = analysisData.score;
  document.getElementById('scoreBig').textContent = analysisData.score;

  const scoreGauge = document.getElementById('scoreGauge');
  if (scoreGauge) {
    const degrees = (analysisData.score / 100) * 360;
    scoreGauge.style.background = `conic-gradient(var(--accent-blue) 0deg ${degrees}deg, var(--border-color) ${degrees}deg 360deg)`;
  }

  // Update Technical Details
  const techGrid = document.getElementById('techGrid');
  techGrid.innerHTML = Object.entries(analysisData.technical)
    .map(([label, value]) => `
      <div class="entity-item">
        <span class="entity-label">${label}:</span>
        <span class="entity-value">${value}</span>
      </div>
    `).join('');

  // Update Success Areas
  const successList = document.getElementById('successList');
  successList.innerHTML = analysisData.success
    .map(item => `<li><span class="issue-icon" style="color: var(--accent-green)">✅</span> ${item}</li>`)
    .join('');

  // Update Issues
  const issuesList = document.getElementById('issuesList');
  issuesList.innerHTML = analysisData.issues
    .map(item => `<li><span class="issue-icon" style="color: var(--accent-red)">❌</span> ${item}</li>`)
    .join('');

  // Update Recommendations
  const recommendationsList = document.getElementById('recommendationsList');
  recommendationsList.innerHTML = analysisData.recommendations
    .map((item, i) => `
      <div class="entity-item" style="margin-bottom: 8px; border-left: 3px solid var(--accent-blue);">
        <span class="entity-label">Suggestion ${i + 1}</span>
        <span class="entity-value">${item}</span>
      </div>
    `).join('');
  // Add to history
  saveToHistory({
    url: analysisData.url,
    score: analysisData.score,
    timestamp: analysisData.timestamp,
  });

  // Increment local scan count for anonymous users
  incrementLocalScan();
}

// State Management
function setState(newState) {
  currentState = newState;

  // Hide all states
  idleState.classList.remove('active');
  analyzingState.classList.remove('active');
  resultsState.classList.remove('active');

  // Show current state
  switch (newState) {
    case 'idle':
      idleState.classList.add('active');
      break;
    case 'analyzing':
      analyzingState.classList.add('active');
      break;
    case 'results':
      resultsState.classList.add('active');
      break;
  }
}

// Panel Management
function showPanel() {
  panelOverlay.classList.remove('hidden');
  document.body.style.overflow = 'hidden';
}

function hidePanel() {
  panelOverlay.classList.add('hidden');
  document.body.style.overflow = 'auto';
}

// Reset Analysis
function resetAnalysis() {
  hidePanel();
  analysisData = null;
  citationStatus.classList.add('hidden');
  setState('idle');
}

// History Modal
function showHistoryModal() {
  historyModal.classList.remove('hidden');
  document.body.style.overflow = 'hidden';
  updateHistoryDisplay();
}

function hideHistoryModal() {
  historyModal.classList.add('hidden');
  document.body.style.overflow = 'auto';
}

// History Management
function saveToHistory(entry) {
  // Store only URL, Date, and AEO Score
  const historyEntry = {
    url: entry.url,
    timestamp: entry.timestamp, // Timestamp in milliseconds
    score: entry.score,
  };

  analysisHistory.unshift(historyEntry);
  if (analysisHistory.length > 20) {
    analysisHistory.pop();
  }

  // Save to chrome.storage.local
  if (chrome && chrome.storage) {
    chrome.storage.local.set({ aeoHistory: analysisHistory });
  }
}

function loadHistoryFromStorage() {
  if (chrome && chrome.storage) {
    chrome.storage.local.get(['aeoHistory'], (result) => {
      if (result.aeoHistory) {
        analysisHistory = result.aeoHistory;
      }
    });
  }
}

function updateHistoryDisplay() {
  const historyList = document.getElementById('historyList');

  if (analysisHistory.length === 0) {
    historyList.innerHTML =
      '<p style="text-align: center; color: #999; padding: 24px;">No analysis history yet</p>';
    return;
  }

  historyList.innerHTML = analysisHistory
    .map(
      (entry) => {
        // Format date: Jan 31, <br> 2026
        const date = new Date(entry.timestamp);
        const monthDay = date.toLocaleDateString('en-US', {
          month: 'short',
          day: 'numeric',
        });
        const year = date.getFullYear();
        const formattedDate = `${monthDay},<br>${year}`;

        return `
    <div class="history-item">
      <span class="history-url" title="${entry.url}">${entry.url}</span>
      <span class="history-date">${formattedDate}</span>
      <span class="history-score">${entry.score}</span>
    </div>
  `;
      }
    )
    .join('');
}

function clearHistory() {
  if (confirm('Are you sure you want to clear all history?')) {
    analysisHistory = [];
    if (chrome && chrome.storage) {
      chrome.storage.local.remove('aeoHistory');
    }
    updateHistoryDisplay();
    hideHistoryModal();
  }
}

// Settings
function openSettings() {
  // Placeholder for settings panel
  console.log('Settings clicked');
  // In a real extension, this would open a detailed settings page
}

// Citation Generation
function generateCitation() {
  if (!analysisData || !analysisData.bestFragment) {
    console.error('[AEO] No analysis data for citation');
    return;
  }

  // Generate Scroll-to-Text fragment
  // 1. Clean the base URL (strip existing # fragments to prevent double-hashes)
  const baseUrl = analysisData.url.split('#')[0];

  // 2. Encode the text fragment
  // Take first 60 chars (shorter is more reliable for matching)
  // Remove special characters that might break fragments
  const cleanFragment = analysisData.bestFragment
    .trim()
    .substring(0, 60)
    .replace(/[#&:]/g, '');

  const citationUrl = `${baseUrl}#:~:text=${encodeURIComponent(cleanFragment)}`;

  // Copy to clipboard
  navigator.clipboard.writeText(citationUrl).then(() => {
    citationStatus.classList.remove('hidden');
    setTimeout(() => {
      citationStatus.classList.add('hidden');
    }, 2000);
  });
}

// Usage Management
async function checkUsage() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['auth_token', 'anonymous_usage'], (result) => {
      // Case 1: User is logged in - Check server
      if (result.auth_token) {
        chrome.runtime.sendMessage({
          action: 'checkUsage',
          token: result.auth_token
        }, (response) => {
          if (response && response.success !== undefined) {
            updateUsageUI(response);
            resolve(response);
          } else {
            resolve({ success: false, error: 'SERVER_ERROR' });
          }
        });
        return;
      }

      // Case 2: Anonymous user - Check local storage
      const now = Date.now();
      let usage = result.anonymous_usage || { used: 0, last_reset: now, limit: 3, plan: 'Free' };

      // 24h Reset logic for local
      if (now - usage.last_reset > 24 * 60 * 60 * 1000) {
        usage.used = 0;
        usage.last_reset = now;
        chrome.storage.local.set({ anonymous_usage: usage });
      }

      const response = {
        success: usage.used < usage.limit,
        used: usage.used,
        limit: usage.limit,
        plan: 'Free (No Login)',
        error: usage.used >= usage.limit ? 'LIMIT_EXCEEDED' : null
      };

      updateUsageUI(response);
      resolve(response);
    });
  });
}

function incrementLocalScan() {
  chrome.storage.local.get(['auth_token', 'anonymous_usage'], (result) => {
    if (result.auth_token) return; // Server handles logged in users

    let usage = result.anonymous_usage || { used: 0, last_reset: Date.now(), limit: 3 };
    usage.used++;
    chrome.storage.local.set({ anonymous_usage: usage }, () => {
      updateUsageUI({
        used: usage.used,
        limit: usage.limit,
        plan: 'Free (No Login)'
      });
    });
  });
}

function updateUsageUI(data) {
  if (!data) {
    // Initial load from storage if data not provided
    chrome.storage.local.get(['last_usage'], (result) => {
      if (result.last_usage) {
        displayUsage(result.last_usage);
      }
    });
    return;
  }

  // Save for offline/faster loading
  chrome.storage.local.set({ last_usage: data });
  displayUsage(data);
}

function displayUsage(data) {
  if (!usageContainer) return;

  usageContainer.classList.remove('hidden');
  const remaining = Math.max(0, data.limit - data.used);

  if (data.limit === Infinity) {
    usageText.textContent = `Scans: Unlimited`;
    usageBarFill.style.width = '100%';
  } else {
    usageText.textContent = `Scans: ${remaining} of ${data.limit} remaining today`;
    const percentage = (data.used / data.limit) * 100;
    usageBarFill.style.width = `${Math.min(100, percentage)}%`;
  }

  planBadge.textContent = data.plan || 'Free';

  if (remaining <= 0 && data.limit !== Infinity) {
    showPaywall(data);
  } else {
    hidePaywall();
  }
}

function showPaywall(data) {
  analyzeBtn.disabled = true;
  analyzeBtn.style.opacity = '0.5';
  paywallContainer.classList.remove('hidden');
}

function hidePaywall() {
  analyzeBtn.disabled = false;
  analyzeBtn.style.opacity = '1';
  paywallContainer.classList.add('hidden');
}
