// Content Script
// This script runs on the page being analyzed
// It extracts page content and communicates with the background script

console.log('[AEO] Content script loaded');

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'analyzeContent') {
    const pageContent = extractPageContent();
    sendResponse({
      success: true,
      mainContent: pageContent.mainContent,
      headings: pageContent.headings,
      paragraphs: pageContent.paragraphs,
      language: pageContent.language,
      url: window.location.href,
      title: document.title,
      bestFragment: pageContent.paragraphs[0] || document.title,
      keywords: pageContent.keywords,
      confusingSentences: pageContent.confusingSentences,
      technical: pageContent.technical
    });
  }
});

// Extract rendered page content
function extractPageContent() {
  // Remove script, style, and other non-content elements
  const clone = document.documentElement.cloneNode(true);
  const selector = 'script, style, noscript, meta, link, [style*="display: none"]';
  clone.querySelectorAll(selector).forEach((el) => el.remove());

  // Get text content
  const text = clone.innerText || clone.textContent || '';

  // Get main content areas
  const mainContent = extractMainContent();
  const paragraphs = getParagraphs(clone);

  // Technical Data
  const h1Count = document.querySelectorAll('h1').length;
  const h2Count = document.querySelectorAll('h2').length;
  const hasSchema = !!document.querySelector('script[type="application/ld+json"]');
  const hasFAQ = isFAQPresent(text);
  const metaDesc = document.querySelector('meta[name="description"]')?.content || '';

  return {
    fullText: text.trim(),
    mainContent: mainContent,
    language: document.documentElement.lang || 'en',
    paragraphs: paragraphs,
    headings: getHeadings(clone),
    keywords: extractKeywords(text),
    confusingSentences: detectConfusingSentences(paragraphs),
    technical: {
      h1Count,
      h2Count,
      hasSchema,
      hasFAQ,
      metaDesc,
      title: document.title,
      firstParaLength: paragraphs[0]?.length || 0
    }
  };
}

// Heuristic to find potential entities (proper nouns/brands)
function extractKeywords(text) {
  const words = text.match(/[A-Z][a-z]{3,}/g) || [];
  const freq = {};
  words.forEach(w => freq[w] = (freq[w] || 0) + 1);
  return Object.entries(freq)
    .sort((a, b) => b[1] - a[1])
    .slice(0, 4)
    .map(entry => entry[0]);
}

// Heuristic to detect vague/confusing sentences
function detectConfusingSentences(paragraphs) {
  const vaguePatterns = [
    /\b(it|this|that) (is|seems|should)\b/i,
    /\b(maybe|perhaps|possibly)\b/i,
    /\b(advanced|expert) knowledge\b/i
  ];

  const results = [];
  paragraphs.forEach((p, i) => {
    if (vaguePatterns.some(pattern => pattern.test(p))) {
      results.push(`Para ${i + 1}: ${p.substring(0, 60)}...`);
    }
  });
  return results.slice(0, 3);
}

// Comprehensive FAQ detection
function isFAQPresent(text) {
  // 1. Check for JSON-LD Schema (FAQPage)
  const scripts = document.querySelectorAll('script[type="application/ld+json"]');
  for (const script of scripts) {
    try {
      if (script.textContent.includes('FAQPage')) return true;
    } catch (e) { }
  }

  // 2. Check for Semantic HTML (details/summary)
  if (document.querySelector('details') && (document.querySelector('summary')?.innerText.match(/\?$/))) {
    return true;
  }

  // 3. Check for common UI patterns (Accordions)
  const faqSelectors = ['.faq', '.accordion', '[id*="faq"]', '[class*="faq"]', '.toggle', '.collapse'];
  for (const selector of faqSelectors) {
    const el = document.querySelector(selector);
    if (el && (el.innerText.toLowerCase().includes('question') || el.innerText.includes('?'))) {
      return true;
    }
  }

  // 4. Check for Heading Pattern (H2/H3 ending in ?)
  const headings = document.querySelectorAll('h2, h3, h4');
  for (const h of headings) {
    if (h.innerText.trim().endsWith('?')) {
      const next = h.nextElementSibling;
      if (next && (next.tagName === 'P' || next.tagName === 'DIV')) return true;
    }
  }

  // 5. Fallback keyword check
  return text.toLowerCase().includes('frequently asked questions');
}

// Extract main content (article, main, or largest text block)
function extractMainContent() {
  const mainElement = document.querySelector('main');
  if (mainElement) {
    return mainElement.innerText || mainElement.textContent || '';
  }

  const articleElement = document.querySelector('article');
  if (articleElement) {
    return articleElement.innerText || articleElement.textContent || '';
  }

  // Fallback: use body
  return document.body.innerText || document.body.textContent || '';
}

// Extract paragraphs
function getParagraphs(element) {
  const paragraphs = [];
  element.querySelectorAll('p').forEach((p) => {
    const text = p.innerText?.trim() || '';
    if (text.length > 20) {
      paragraphs.push(text);
    }
  });
  return paragraphs;
}

// Extract headings
function getHeadings(element) {
  const headings = [];
  element.querySelectorAll('h1, h2, h3, h4, h5, h6').forEach((h) => {
    const text = h.innerText?.trim() || '';
    if (text.length > 0) {
      headings.push({
        level: h.tagName,
        text: text,
      });
    }
  });
  return headings;
}

// Inject analytics or tracking (if needed)
function initializeExtension() {
  // Add visual indicator that extension is active
  const indicator = document.createElement('div');
  indicator.id = 'aeo-extension-indicator';
  indicator.style.cssText = `
    position: fixed;
    bottom: 20px;
    right: 20px;
    width: 12px;
    height: 12px;
    border-radius: 50%;
    background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
    opacity: 0.5;
    z-index: 999999;
    pointer-events: none;
  `;
  document.body.appendChild(indicator);
}

// Initialize on page load
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initializeExtension);
} else {
  initializeExtension();
}
