// Background Service Worker

// Chrome is globally available in background scripts, no need to declare it


// Listen for extension icon click
chrome.action.onClicked.addListener((tab) => {
  console.log('[AEO] Extension icon clicked on tab:', tab.id);
});

// Handle messages from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('[AEO] Message received:', request);

  if (request.action === 'analyzeContent') {
    // Process analysis
    processAnalysis(request.data).then((result) => {
      sendResponse({ success: true, result });
    });
    return true; // Keep connection open for async response
  }

  if (request.action === 'saveAnalysis') {
    saveAnalysisToStorage(request.data);
    sendResponse({ success: true });
  }

  // Handle Auth Sync from website
  if (request.action === 'auth_sync') {
    chrome.storage.local.set({
      auth_token: request.token,
      uid: request.uid,
      user_email: request.email
    }, () => {
      console.log('[AEO] Auth synced for:', request.email);
      sendResponse({ success: true });
    });
    return true;
  }

  if (request.action === 'auth_logout') {
    chrome.storage.local.remove(['auth_token', 'uid', 'user_email'], () => {
      console.log('[AEO] User logged out');
      sendResponse({ success: true });
    });
    return true;
  }

  // Handle usage validation before analysis
  if (request.action === 'checkUsage') {
    validateUsage(request.token).then(sendResponse);
    return true;
  }
});

// Process analysis data
async function processAnalysis(data) {
  // This is where we would run actual analysis logic
  // For now, returning mock data

  return {
    score: calculateScore(data),
    timestamp: new Date().toISOString(),
    url: data.url,
  };
}

// Calculate AEO Score (mock implementation)
function calculateScore(data) {
  // In a real extension, this would analyze:
  // - Q&A readiness
  // - Answer blocks
  // - Entity clarity
  // - Information density
  // - Language simplicity

  return Math.floor(Math.random() * 40) + 40; // Random 40-80 for demo
}

// Storage helper
function saveAnalysisToStorage(data) {
  chrome.storage.local.get(['analyses'], (result) => {
    const analyses = result.analyses || [];
    analyses.unshift({
      ...data,
      savedAt: new Date().toISOString(),
    });

    // Keep only last 50 analyses
    if (analyses.length > 50) {
      analyses.pop();
    }

    chrome.storage.local.set({ analyses });
  });
}

// Validate Usage with Backend
async function validateUsage(token) {
  if (!token) {
    return { success: false, error: 'NOT_LOGGED_IN' };
  }

  try {
    // Replace with actual Firebase Cloud Function URL
    const functionUrl = 'https://YOUR_REGION-YOUR_PROJECT.cloudfunctions.net/validateAndIncrementScan';

    // In a real implementation, you'd use the Firebase JS SDK or a fetch call with the Bearer token
    // For this example, we'll use fetch to represent the interaction
    const response = await fetch(functionUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({ data: {} }) // Firebase callable functions expect { data: ... }
    });

    const result = await response.json();
    return result.result; // Firebase callable returns { result: { success, ... } }
  } catch (error) {
    console.error('[AEO] Usage check failed:', error);
    return { success: false, error: 'SERVER_ERROR' };
  }
}

// Handle tab updates
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete') {
    console.log('[AEO] Tab loaded:', tab.url);
  }
});
